"use client";

import * as React from "react";
import { useParams } from "next/navigation";
import { useTranslations } from "next-intl";

type POData = {
  id: string;
  poNumber: string;
  supplierId: string;
  supplierEmail: string;
  supplierPhone: string;
  poDate: string;
  expectedDate: string;
  status: "draft" | "sent" | "partial" | "closed";
  currency: string;
  terms: string;
  deliverTo: string;
  notes: string;
  lines: any[];
  charges: {
    shipping: number;
    other: number;
  };
};

function getSamplePO(id: string): POData {
  return {
    id,
    poNumber: "PO-000123",
    supplierId: "s1",
    supplierEmail: "sales@acme.com",
    supplierPhone: "+1 555 123",
    poDate: "2025-08-01",
    expectedDate: "2025-08-05",
    status: "draft",
    currency: "QAR",
    terms: "net30",
    deliverTo: "main",
    notes: "Please deliver ASAP.",
    lines: [
      {
        id: "1",
        item: "i1",
        description: "Latex",
        unit: "box",
        qty: 10,
        price: 12.5,
        discount: 0,
        tax: 5,
      },
    ],
    charges: {
      shipping: 10,
      other: 5,
    },
  };
}

export default function ViewPurchaseOrderPage() {
  const { id } = useParams();
  const t = useTranslations("PurchaseOrders");

  const po = getSamplePO(id as string);
  const total = po.lines.reduce((acc, line) => {
    const base = line.qty * line.price;
    const afterDisc = base - (line.discount / 100) * base;
    const afterTax = afterDisc + (line.tax / 100) * afterDisc;
    return acc + afterTax;
  }, 0) + po.charges.shipping + po.charges.other;

  return (
    <div className="p-6 space-y-6">
      <h1 className="text-2xl font-semibold">
        {t("poNumber")}: {po.poNumber}
      </h1>

      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
        <div><strong>{t("supplier")}</strong>: {po.supplierId}</div>
        <div><strong>{t("poDate")}</strong>: {po.poDate}</div>
        <div><strong>{t("expectedDate")}</strong>: {po.expectedDate}</div>
        <div><strong>{t("status")}</strong>: {t(`status${po.status.charAt(0).toUpperCase() + po.status.slice(1)}`)}</div>
        <div><strong>{t("currency")}</strong>: {po.currency}</div>
        <div><strong>{t("terms")}</strong>: {po.terms}</div>
        <div><strong>{t("deliverTo")}</strong>: {po.deliverTo}</div>
        <div><strong>{t("supplierEmail")}</strong>: {po.supplierEmail}</div>
        <div><strong>{t("supplierPhone")}</strong>: {po.supplierPhone}</div>
      </div>

      <div>
        <h2 className="text-lg font-semibold">{t("itemsSection")}</h2>
        <table className="w-full border mt-2 text-sm">
          <thead className="bg-muted">
            <tr>
              <th className="p-2 text-left">{t("item")}</th>
              <th className="p-2 text-left">{t("description")}</th>
              <th className="p-2">{t("qty")}</th>
              <th className="p-2">{t("unitPrice")}</th>
              <th className="p-2">{t("discount")}</th>
              <th className="p-2">{t("tax")}</th>
              <th className="p-2 text-right">{t("lineTotal")}</th>
            </tr>
          </thead>
          <tbody>
            {po.lines.map((line) => {
              const base = line.qty * line.price;
              const afterDisc = base - (line.discount / 100) * base;
              const lineTotal = afterDisc + (line.tax / 100) * afterDisc;

              return (
                <tr key={line.id} className="border-t">
                  <td className="p-2">{line.item}</td>
                  <td className="p-2">{line.description}</td>
                  <td className="p-2 text-center">{line.qty}</td>
                  <td className="p-2 text-center">{line.price}</td>
                  <td className="p-2 text-center">{line.discount}%</td>
                  <td className="p-2 text-center">{line.tax}%</td>
                  <td className="p-2 text-right">{lineTotal.toFixed(2)}</td>
                </tr>
              );
            })}
          </tbody>
        </table>
      </div>

      <div className="text-right space-y-1">
        <div>{t("shipping")}: {po.charges.shipping.toFixed(2)} {po.currency}</div>
        <div>{t("otherCharges")}: {po.charges.other.toFixed(2)} {po.currency}</div>
        <div className="text-lg font-semibold">
          {t("grandTotal")}: {total.toFixed(2)} {po.currency}
        </div>
      </div>

      <div>
        <h3 className="font-medium">{t("notes")}</h3>
        <p className="text-muted-foreground">{po.notes}</p>
      </div>
    </div>
  );
}
